<?php
/**
 * Copyright (c) 2016 Serhii Borodai <clarifying@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *
 */

namespace App\Action;


use App\Service\Mailer;
use App\Entity\Feedback;
use App\Validator\Feedback as FeedbackValidator;
use Interop\Container\ContainerInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\HtmlResponse;
use Zend\Diactoros\Response\JsonResponse;
use Zend\Expressive\Router\RouterInterface;
use Zend\Expressive\Template\TemplateRendererInterface;
use Zend\Hydrator\DelegatingHydrator;
use Zend\Validator\ValidatorInterface;

/**
 * Class Contacts
 * @package App\Action
 */
class Contacts extends Common
{
    /**
     * @var Mailer
     */
    protected $mailer;
    /**
     * @var DelegatingHydrator
     */
    protected $hydrator;

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param callable|null $next
     * @return HtmlResponse
     */
    function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null)
    {
        $this->initInfo($request);
        if ($this->isXmlHttpRequest($request)) {

            $postData = $request->getParsedBody();
            $feedback = $this->hydrator->hydrate($postData, new Feedback());

            /** @var ValidatorInterface $validator */
            $validator = new FeedbackValidator($this->hydrator);
            if ($validator->isValid($feedback)) {
                $this->mailer->sendFeedback($feedback);
                $result = true;
                $messages = [];
            } else {
                $result = false;
                $messages = $validator->getMessages();
            }

            $response = new JsonResponse(
                [
                    'result' => $result,
                    'msg' => $messages,
                ]
            );
        } else {
            $response = new HtmlResponse($this->template->render('app::contacts', [
                'lang' => $request->getAttribute('layoutInfo')->getLang(),
                'layoutInfo' => $request->getAttribute('layoutInfo'),
            ]));
        }

        return $response;
    }

    /**
     * Contacts constructor.
     * @param RouterInterface $router
     * @param null|TemplateRendererInterface $template
     * @param ContainerInterface $container
     */
    public function __construct(RouterInterface $router, $template = null, ContainerInterface $container)
    {
        parent::__construct($router, $template, $container);
        $this->mailer = $this->container->get(Mailer::class);
        $this->hydrator = $this->container->get(DelegatingHydrator::class);
    }


}