<?php
/**
 * Copyright (c) 2016 Serhii Borodai <clarifying@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *
 */

namespace App\Action\User;

use App\Action\Common;
use App\Authentication\UserService;
use App\Model\Users;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\HtmlResponse;
use Zend\Diactoros\Response\JsonResponse;
use Zend\Diactoros\Response\RedirectResponse;
use Zend\Hydrator\DelegatingHydrator;

/**
 * Class CabinetAmSystem
 * @package App\Action\User
 */
class CabinetAmSystem extends Common
{
    
    const ACTION_GROUPS_LIST = 'groups';
    const ACTION_GROUPS_ITEM = 'groups_item';
    const ACTION_CAMPAIGNS_ITEM = 'campaigns_item';
    const ACTION_GROUPS_STAT = 'groups_stat';
    const ACTION_CAMPAIGNS_STAT = 'campaigns_stat';
    
    const CAMPAIGN_STAT_CATS = [
        'views'         =>  ['title' => 'Показы',   'color' => '#2CA02C'],
        'clicks'        =>  ['title' => 'Клики',    'color' => '#D62728'],
        'reach'         =>  ['title' => 'Охват',    'color' => '#FF7F0E'],
        'ctr'           =>  ['title' => 'CTR',      'color' => '#9467BD'],
        'cpm'           =>  ['title' => 'eCPM',     'color' => '#33C7B6'],
        'cpc'           =>  ['title' => 'eCPC',     'color' => '#7CADE2'],
        'money'         =>  ['title' => 'Расход',   'color' => '#1F77B4'],
        'order_cnt'     =>  ['title' => 'Заказы',   'color' => '#A4BA00'],
        'order_money'   =>  ['title' => 'Доход',   'color' => '#BA3A9E'],
    ];
    const IMG_DOMAIN_DEV = 'https://addcpm.app.dev.rew.to';
    const IMG_DOMAIN_PROD = 'https://www.addcpm.com';

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param callable|null $next
     * @return HtmlResponse
     */
    function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null)
    { 
        
        if($request->getMethod() == 'POST') {
            
            try {
                switch ($request->getAttribute('action')) {
                    case self::ACTION_GROUPS_STAT:
                        $response = $this->getGroupsStatData($request);
                        break;
                    case self::ACTION_CAMPAIGNS_STAT:
                        $response = $this->getCampaignsStatData($request);
                        break;
                    default:                    
                }
            } catch(\Exception $e) {
                $data = [
                    'result' => false,
                    'msg' => $e->getMessage(),
                ];
                $response = new JsonResponse($data);
            }
            return $response;
            
        }
        elseif($request->getAttribute('action') == self::ACTION_GROUPS_LIST)
        {
            
            try {

                /** @var UserService $auth */
                $auth = $this->container->get(UserService::class);
                $userId = $auth->getIdentity()->getId();
                
                /** @var \App\Model\AmSystem\Groups $groupsModel */
                $groupsModel = $this->container->get(\App\Model\AmSystem\Groups::class);
                $groups = $groupsModel->findAll(['user_id' => $userId])->toArray();
                
                if ($groups) {
                    
                    /** @var \App\Model\AmSystem\Campaigns $campaignsModel */
                    $campaignsModel = $this->container->get(\App\Model\AmSystem\Campaigns::class);
                    
                    /** @var \App\Model\Statistics $stats */
                    $stats = $this->container->get(\App\Model\Statistics::class);
                    $groups_stat = [];
                    $campaigns_stat = [];
                    
                    $groups_campaigns = [];
                    foreach($groups as $group) {
                        $campaigns = $campaignsModel->findAll(['group_id' => $group['id']])->toArray();
                        $groups_campaigns[$group['id']] = $campaigns;
                    
                        $group_id = $group['id'];
                        $groups_stat[$group_id] = $stats->getCampaignsStat('group', [$group_id], $group['target_action'], null, false);
                        
                        foreach ($campaigns as $campaign_item) {
                            $campaign_id = $campaign_item['id'];
                            $campaigns_stat[$campaign_id] = $stats->getCampaignsStat('campaign', [$campaign_id], $campaign_item['target_action'], null, false);
                        }
                    }
                    
                    $data['groups'] = $groups;
                    $data['campaigns'] = $campaigns;
                    $data['groups_campaigns'] = $groups_campaigns;
                    $data['groups_stat'] = $groups_stat;
                    $data['campaigns_stat'] = $campaigns_stat;
                    $data['stat_cats'] = self::CAMPAIGN_STAT_CATS;
                    
                } else {
                    $data['error'] = _t('Отсутствуют доступные кампании');
                }
                

            } catch(\Exception $e) {
                $data['error'] = _t('Извините, данные по кампаниям временно недоступны');
            }

            $data = array_merge($data, [
                'lang' => $request->getAttribute('layoutInfo')->getLang(),
                'layoutInfo' => $request->getAttribute('layoutInfo'),
            ]);     

            return new HtmlResponse($this->template->render('app::user/amsystem/groups_list', $data));
        }
        elseif($request->getAttribute('action') == self::ACTION_GROUPS_ITEM)
        {
            $group_id = $request->getAttribute('id');
            
            /** @var UserService $auth */
            $auth = $this->container->get(UserService::class);
            $userId = $auth->getIdentity()->getId();
            
            /** @var \App\Model\AmSystem\Groups $groupsModel */
            $groupsModel = $this->container->get(\App\Model\AmSystem\Groups::class);
            $group = $groupsModel->findOne(['id' => $group_id]);
            
            if($group) {
                
                if ($group->getUserId() == $userId) {

                    /** @var \App\Model\AmSystem\Campaigns $campaignsModel */
                    $campaignsModel = $this->container->get(\App\Model\AmSystem\Campaigns::class);
                    $campaigns = $campaignsModel->findAll(['group_id' => $group_id])->toArray();

                    /** @var \App\Model\Statistics $stats */
                    $stats = $this->container->get(\App\Model\Statistics::class);
                    $campaigns_stat = [];

                    $isTrackingCampaign = false;

                    $date_begin = date("Y-m-d");
                    $date_end_common = date('2020-01-01');
                    $date_end = date('2020-01-01');

                    foreach ($campaigns as $campaign_item) {

                        if ($campaign_item['tracking_action']) {
                            $isTrackingCampaign = true;
                        }


                        $campaign_id = $campaign_item['id'];
                        $campaigns_stat[$campaign_id] = $stats->getCampaignsStat('campaign', [$campaign_id], $campaign_item['target_action'], null, false);

                        $campaign_begin = $campaign_item['date_begin'];
                        $campaign_end = $campaign_item['date_end'];

                        if ($campaign_begin && $campaign_begin < $date_begin) {
                            $date_begin = $campaign_begin;
                        }
                        if ($campaign_end && $campaign_end > $date_end) {
                            $date_end = $campaign_end;
                        }
                        if (!$campaign_end) {
                            $date_end_common = null;
                        }
                    }
                    
                    $date_begin = date("Y-m-d", strtotime($date_begin));
                    $date_end = date("Y-m-d", strtotime($date_end));
                    
                    if ($date_end_common) {
                        $date_end_common = $date_end;
                    }
                    

                    /** @var \App\Model\Users $userModel */
                    $userModel = $this->container->get(\App\Model\Users::class);
                    $users_list = $userModel->findAll(['type' => 'advertiser']);

                    $users = [];
                    foreach($users_list as $user_item) {
                        $users[$user_item->getId()] = [$user_item->getFullname(), $user_item->getEmail()];
                    }

                    /** @var \App\Model\Statistics $stats */
                    $stats = $this->container->get(\App\Model\Statistics::class);
                    $dates_common = [$date_begin, $date_end_common];
                    $dates = [$date_begin, $date_end];
                    $group_stat = $stats->getCampaignsStat('group', [(int)$group_id], $group->getTargetAction(), $dates_common, true);
                    $group_stat_total = $stats->getCampaignsStat('group', [(int)$group_id], $group->getTargetAction(), $dates_common, false);
                    $sites_stats = $stats->getCampaignsSitesStat('group', (int)$group_id, $group->getTargetAction(), $dates);
                    $regions_stats = $stats->getCampaignsRegionsStat('group', (int)$group_id, $group->getTargetAction(), $dates);
                    $regions_stats = $this->prepareRegionsData($regions_stats);

                    // Добавляем размер зоны в данные по сайтам:
                    /** @var \App\Model\Zones $zonesModel */
                    $zonesModel = $this->container->get(\App\Model\Zones::class);
                    if ($sites_stats && !($sites_stats['status'] && $sites_stats['status'] == 'Error')) {
                        foreach ($sites_stats as $site_name => $site_data) {
                            if ($site_data['sub']) {
                                foreach ($site_data['sub'] as $zone_id => $zone_data) {
                                    $zone = $zonesModel->findById($zone_id);
                                    if ($zone) {
                                        $zone_width = $zone->getWidth();
                                        $zone_height = $zone->getHeight();
                                        $zone_size = $zone_width.'x'.$zone_height;
                                    } else {
                                        $zone_size = null;
                                    }
                                    $sites_stats[$site_name]['sub'][$zone_id]['size'] = $zone_size;
                                }
                            }
                        }
                    }

                    $stat_cats = self::CAMPAIGN_STAT_CATS;
                    if (!$isTrackingCampaign) {
                        unset($stat_cats['order_cnt']);
                        unset($stat_cats['order_money']);
                    }

                    
                    $data['lang'] = $request->getAttribute('layoutInfo')->getLang();
                    $data['countries'] = $this->container->get('config')['amsystem_conf']['country'];
                    $data['regions'] = $this->container->get('config')['amsystem_conf']['region'];
                    $data['cities'] = $this->container->get('config')['amsystem_conf']['city'];
                    $data['users'] = $users;
                    $data['group'] = $group;
                    $data['campaigns'] = $campaigns;
                    $data['campaigns_stat'] = $campaigns_stat;
                    $data['sites_stats'] = $sites_stats;
                    $data['regions_stats'] = $regions_stats;
                    $data['group_stat'] = $group_stat;
                    $data['group_stat_total'] = $group_stat_total;
                    $data['stat_cats'] = $stat_cats;
                    $data['active_line'] = 'views';
                    $data['group_date_begin'] = $date_begin;
                    $data['group_date_end'] = $date_end;
                    $data['code_regions'] = $this->getCodeRegions();
                    $data['code_cities'] = $this->getCodeCities();
                    
                } else {
                    $data['error'] = _t('Извините, данная группа закреплена за другим пользователем');
                }
            } else {
                $data['error'] = _t('Извините, группа не найдена');
            }
            
            $data = array_merge($data, [
                'lang' => $request->getAttribute('layoutInfo')->getLang(),
                'layoutInfo' => $request->getAttribute('layoutInfo'),
            ]);     

            return new HtmlResponse($this->template->render('app::user/amsystem/group_item', $data));
        }
        elseif($request->getAttribute('action') == self::ACTION_CAMPAIGNS_ITEM)
        {
            $campaign_id = $request->getAttribute('id');
            
            /** @var UserService $auth */
            $auth = $this->container->get(UserService::class);
            $userId = $auth->getIdentity()->getId();
            
            /** @var \App\Model\AmSystem\Groups $groupsModel */
            $groupsModel = $this->container->get(\App\Model\AmSystem\Groups::class);
            
            /** @var \App\Model\AmSystem\Campaigns $campaignsModel */
            $campaignsModel = $this->container->get(\App\Model\AmSystem\Campaigns::class);
            $campaign = $campaignsModel->findById($campaign_id);
            
            if ($campaign) {

                $group_id = $campaign->getGroupId();
                $group = $groupsModel->findOne(['id' => $group_id]);
                
                if ($group->getUserId() == $userId){
                    

                    /** @var \App\Model\AmSystem\Campaigns $campaignsModel */
                    $campaignsModel = $this->container->get(\App\Model\AmSystem\Campaigns::class);
                    $campaign = $campaignsModel->findOne(['id' => $campaign_id]);
                    if(!$campaign) {
                        return new HtmlResponse($this->template->render('error::404', ['lang' => $request->getAttribute('layoutInfo')->getLang()]));
                    }

                    $group_id = $campaign->getGroupId();

                    /** @var \App\Model\AmSystem\Groups $groupsModel */
                    $groupsModel = $this->container->get(\App\Model\AmSystem\Groups::class);
                    $group = $groupsModel->findOne(['id' => $group_id]);
                    if(!$group) {
                        return new HtmlResponse($this->template->render('error::404', ['lang' => $request->getAttribute('layoutInfo')->getLang()]));                    
                    }

                    /** @var \App\Model\Users $userModel */
                    $userModel = $this->container->get(\App\Model\Users::class);
                    $user = $userModel->findOne(['id' => $group->getUserId()]);

                    /** @var \App\Model\Statistics $stats */
                    $stats = $this->container->get(\App\Model\Statistics::class);

                    $campaign_banners_id = $campaign->getBanners();
                    if ($campaign_banners_id) {
                        $banners_id = json_decode($campaign_banners_id);

                        /** @var \App\Model\AmSystem\Banners $bannersModel */
                        $bannersModel = $this->container->get(\App\Model\AmSystem\Banners::class);
                        $banners = $bannersModel->findAll(['id' => $banners_id])->toArray();

                        $banners_stats = [];
                        foreach ($banners as $banner_item) {
                            $banner_id = $banner_item['id'];
                            $banners_stats[$banner_id] = $stats->getBannersStat([$banner_id], $campaign->getTargetAction(), null);
                        }
                    }

                    $date_begin = date("Y-m-d", strtotime($campaign->getDateBegin()));
                    $date_end_common = $campaign->getDateEnd() ? (date("Y-m-d", strtotime($campaign->getDateEnd()))): null;
                    $date_end = $campaign->getDateEnd() ? date("Y-m-d", strtotime($campaign->getDateEnd())) : date("Y-m-d");
                    $dates_common = [$date_begin, $date_end_common];
                    $dates = [$date_begin, $date_end];
                    $campaign_stat = $stats->getCampaignsStat('campaign', [(int)$campaign_id], $campaign->getTargetAction(), $dates_common, true);
                    $campaign_stat_total = $stats->getCampaignsStat('campaign', [(int)$campaign_id], $campaign->getTargetAction(), $dates_common, false);
                    $sites_stats = $stats->getCampaignsSitesStat('campaign', (int)$campaign_id, $campaign->getTargetAction(), $dates);
                    $regions_stats = $stats->getCampaignsRegionsStat('campaign', (int)$campaign_id, $campaign->getTargetAction(), $dates);
                    $regions_stats = $this->prepareRegionsData($regions_stats);

                    // Добавляем размер зоны в данные по сайтам:
                    /** @var \App\Model\Zones $zonesModel */
                    $zonesModel = $this->container->get(\App\Model\Zones::class);

                    if ($sites_stats && !($sites_stats['status'] && $sites_stats['status'] == 'Error')) {
                        foreach ($sites_stats as $site_name => $site_data) {
                            if ($site_data['sub']) {
                                foreach ($site_data['sub'] as $zone_id => $zone_data) {
                                    $zone = $zonesModel->findById($zone_id);
                                    if ($zone) {
                                        $zone_width = $zone->getWidth();
                                        $zone_height = $zone->getHeight();
                                        $zone_size = $zone_width.'x'.$zone_height;
                                    } else {
                                        $zone_size = null;
                                    }
                                    $sites_stats[$site_name]['sub'][$zone_id]['size'] = $zone_size;
                                }
                            }
                        }
                    }

                    $active_line = 'views';

                    $stat_cats = self::CAMPAIGN_STAT_CATS;
                    if (!$campaign->getTrackingAction()) {
                        unset($stat_cats['order_cnt']);
                        unset($stat_cats['order_money']);
                    }


                    $data['lang'] = $request->getAttribute('layoutInfo')->getLang();
                    $data['campaign'] = $campaign;
                    $data['countries'] = $this->container->get('config')['amsystem_conf']['country'];
                    $data['regions'] = $this->container->get('config')['amsystem_conf']['region'];
                    $data['cities'] = $this->container->get('config')['amsystem_conf']['city'];
                    $data['sites_stats'] = $sites_stats;
                    $data['regions_stats'] = $regions_stats;
                    $data['campaign_stat'] = $campaign_stat;
                    $data['campaign_stat_total'] = $campaign_stat_total;
                    $data['stat_cats'] = $stat_cats;
                    $data['active_line'] = $active_line;
                    $data['banners'] = $banners;
                    $data['banners_stats'] = $banners_stats;
                    $data['group'] = $group;
                    $data['user'] = $user;
                    $data['code_regions'] = $this->getCodeRegions();
                    $data['code_cities'] = $this->getCodeCities();
                    $data['img_domain'] = (getenv('DEVELOPMENT')) ? self::IMG_DOMAIN_DEV : self::IMG_DOMAIN_PROD;
                    $data['dates_common'] = $dates_common;

                } else {
                    $data['error'] = _t('Извините, данная кампания закреплена за другим пользователем');
                }

            } else {
                $data['error'] = _t('Извините, кампания не найдена');
            }
            
            $data = array_merge($data, [
                'lang' => $request->getAttribute('layoutInfo')->getLang(),
                'layoutInfo' => $request->getAttribute('layoutInfo'),
            ]);     

            return new HtmlResponse($this->template->render('app::user/amsystem/campaign_item', $data));
        }
    }
    
    
    private function getGroupsStatData(ServerRequestInterface $request) 
    {
        $stat_data = $this->getAmSystemStatData('group', $request);
        return new JsonResponse($stat_data);
    }
    
    private function getCampaignsStatData(ServerRequestInterface $request) 
    {
        $stat_data = $this->getAmSystemStatData('campaign', $request);
        return new JsonResponse($stat_data);
    }
    
    private function getAmSystemStatData($type, ServerRequestInterface $request)
    {
        $item_id = $request->getAttribute('id');
        
        if ($type == 'campaign') {
            /** @var \App\Model\AmSystem\Campaigns $campaignsModel */
            $campaignsModel = $this->container->get(\App\Model\AmSystem\Campaigns::class);
            $item = $campaignsModel->findById($item_id);
        } else {
            /** @var \App\Model\AmSystem\Groups $groupsModel */
            $groupsModel = $this->container->get(\App\Model\AmSystem\Groups::class);
            $item = $groupsModel->findById($item_id);
        }
        
        if ($item) {
                
            /** @var \App\Model\Statistics $stats */
            $stats = $this->container->get(\App\Model\Statistics::class);

            $data_request = $request->getParsedBody();

            if ($data_request['type'] == 'stat-common') {
                $data_stat = [
                    'stat' => $stats->getCampaignsStat($type, [(int)$item_id], $item->getTargetAction(), $data_request['period'], true),
                    'total' => $stats->getCampaignsStat($type, [(int)$item_id], $item->getTargetAction(), $data_request['period'], false)
                ];
            }

            if ($data_request['type'] == 'stat-sites') {
                $sites_stats = $stats->getCampaignsSitesStat($type, (int)$item_id, $item->getTargetAction(), $data_request['period']);
                
                // Добавляем размер зоны в данные по сайтам:
                /** @var \App\Model\Zones $zonesModel */
                $zonesModel = $this->container->get(\App\Model\Zones::class);
                if ($sites_stats && !($sites_stats['status'] && $sites_stats['status'] == 'Error')) {
                    foreach ($sites_stats as $site_name => $site_data) {
                        if ($site_data['sub']) {
                            foreach ($site_data['sub'] as $zone_id => $zone_data) {
                                $zone = $zonesModel->findById($zone_id);
                                if ($zone) {
                                    $zone_width = $zone->getWidth();
                                    $zone_height = $zone->getHeight();
                                    $zone_size = $zone_width.'x'.$zone_height;
                                } else {
                                    $zone_size = null;
                                }
                                $sites_stats[$site_name]['sub'][$zone_id]['size'] = $zone_size;
                            }
                        }
                    }
                }
                
                $data_stat = $sites_stats;
                
            }

            if ($data_request['type'] == 'stat-regions') {
                $data_stat = $stats->getCampaignsRegionsStat($type, (int)$item_id, $item->getTargetAction(), $data_request['period']);
                $data_stat = $this->prepareRegionsData($data_stat);
            }

            return array(
                'type' => $data_request['type'],
                'data' => $data_stat
            );

        }
        
        return null;
    }
    
    private function prepareRegionsData($data)
    {
        if ($data) {
            
            // Убираем города у регионов Киев(703447), Севастополь(694422) и empty:
            foreach ([703447, 694422, 'empty'] as $region_code) {
                unset($data[$region_code]['sub']);
            }
            
            $code_regions = $this->getCodeRegions();
            $code_cities = $this->getCodeCities();
            
            foreach ($data as $region => $stat) {
                
                if (!$code_regions[$region]) {
                    unset($data[$region]);
                } else {
                    if ($stat['sub']) {
                        foreach ($stat['sub'] as $city => $stat_city) {
                            if (!$code_cities[$city]) {
                                unset($data[$region]['sub'][$city]);
                            }
                        }
                    }
                }
            }
        }
        return $data;
    }
    
    private function getCodeRegions()
    {
        $code_regions = [];
        $regions = $this->container->get('config')['amsystem_conf']['region'];
        foreach ($regions as $region_item) {
            foreach ($region_item['codes'] as $code) {
                $code_regions[$code] = str_replace(' область', '', $region_item['name']);
            }
        }
        return $code_regions;
    }
    
    private function getCodeCities()
    {
        $code_cities = [];
        $cities = $this->container->get('config')['amsystem_conf']['city'];
        foreach ($cities as $city_item) {
            foreach ($city_item['codes'] as $code) {
                $code_cities[$code] = str_replace(' область', '', $city_item['name']);
            }
        }
        return $code_cities;
    }
    
}