<?php
/**
 * Copyright (c) 2016 Serhii Borodai <clarifying@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *
 */

namespace App\Action;


use App\Model\Articles;
use App\Model\Tags;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\HtmlResponse;
use Zend\Diactoros\Response\JsonResponse;
use Zend\Http\Response;

/**
 * Class Blog
 * @package App\Action
 */
class Blog extends Common
{
    const ACTION_LIST = 'blog article list';
    const ACTION_ARTICLE = 'blog article';
    const FORMAT_JSON = 'json';

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param callable|null $next
     * @return ResponseInterface|HtmlResponse
     * @throws \Exception
     */
    function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null)
    {
        $this->initInfo($request);

        switch ($request->getAttribute('action')) {
            case self::ACTION_ARTICLE:
                $articleId = $request->getAttribute('article');
                if ($articleId) {

                    /** @var Articles $articlesModel */
                    $articlesModel = $this->container->get(Articles::class);
                    $article = $articlesModel->findById($articleId);

                    if ($article) {
                        $response = new HtmlResponse($this->template->render("app::blog-article", [
                            'lang' => $request->getAttribute('layoutInfo')->getLang(),
                            'article' => $article,
                        ]));
                    } else {
                        $response->withStatus(Response::STATUS_CODE_404);
                    }
                } else {
                    $response->withStatus(Response::STATUS_CODE_404);
                }

                break;
            case self::ACTION_LIST:
                $page = $request->getParsedBody()['page'] ?? 1;
                $tag = $request->getAttribute('tag', '');
                $perPage = $this->container->get('config')['blog']['articlesPerPage'];

                /** @var Articles $articlesModel */
                $articlesModel = $this->container->get(Articles::class);
                /** @var Tags $tagsModel */
                $tagsModel = $this->container->get(Tags::class);

                $data = $tag ? $articlesModel->findLimitWithTag([
                    Articles::PLACEHOLDER_LIMIT => $perPage,
                    Articles::PLACEHOLDER_OFFSET => ($page - 1) * $perPage,
                ],
                    $tag) : $articlesModel->findLimit([
                    Articles::PLACEHOLDER_LIMIT => $perPage,
                    Articles::PLACEHOLDER_OFFSET => ($page - 1) * $perPage,
                ]);

                $totalArticles = $articlesModel->foundRows();
                $pageNumbers = ceil($totalArticles / $perPage);

                $tags = $tagsModel->findAll();

                if ($request->getAttribute('format') == self::FORMAT_JSON) {
                    $response = new JsonResponse([
                        'data' => $data,
                        'lang' => $request->getAttribute('layoutInfo')->getLang(),
                        'pageNumbers' => $pageNumbers,
                        'currentPage' => $page,
                        'tag' => $tag,
                        'tags' => $tags,
                    ]);
                } else {
                    $response = new HtmlResponse($this->template->render("app::blog-list", [
                        'lang' => $request->getAttribute('layoutInfo')->getLang(),
                        'pageNumbers' => $pageNumbers,
                        'currentPage' => $page,
                        'tag' => $tag,
                        'tags' => $tags,
                    ]));
                }
                break;
            default:
                throw new \Exception('Unknown blog action');
                break;
        }

        return $response;
    }
}