<?php
/**
 * Copyright (c) 2017 Serhii Borodai <clarifying@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *
 */

namespace App\Action\User;


use App\Action\Common;
use App\Authentication\UserService;
use App\Service\Mailer;
use Interop\Container\ContainerInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\HtmlResponse;
use Zend\Diactoros\Response\JsonResponse;
use Zend\Diactoros\Response\RedirectResponse;
use Zend\Expressive\Router\RouterInterface;
use Zend\Expressive\Template\TemplateRendererInterface;

/**
 * Class Mail
 * @package App\Action\User
 */
class Mail extends Common
{
    const SEND_ACTIVATION = 'send activation mail';
    const SEND_PASSWORD = 'send restore password mail';
    const REQUEST_ACTIVATE_PROTECTION = 'send connect request';

    /**
     * @var Mailer
     */
    protected $mailer;
    /**
     * @var UserService
     */
    protected $auth;

    /**
     * Common constructor.
     * @param RouterInterface $router
     * @param TemplateRendererInterface|null $template
     * @param ContainerInterface $container
     */
    public function __construct(RouterInterface $router, $template = null, ContainerInterface $container)
    {
        parent::__construct($router, $template, $container);

        $this->mailer = $container->get(Mailer::class);
        $this->auth = $container->get(UserService::class);
    }

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param callable|null $next
     * @return ResponseInterface|string|RedirectResponse
     */
    function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null)
    {
        $this->initInfo($request);

        $action = $request->getAttribute('action');
        switch ($action) {
            case self::SEND_ACTIVATION:
                $this->sendActivationMail($request);
                $response = new HtmlResponse($this->template->render('app::user/sent-activation', [
                    'lang' => $request->getAttribute('layoutInfo')->getLang(),
                ]));
                break;
            case self::SEND_PASSWORD:
                $response = new HtmlResponse($this->template->render('app::user/sent-password', [
                    'lang' => $request->getAttribute('layoutInfo')->getLang(),
                ]));
                break;
            case self::REQUEST_ACTIVATE_PROTECTION:
                $this->requestActivateProtection($request);
                $response = new JsonResponse([
                    'result' => true,
                    'msg' => '',
                ]);
                break;
            default:
                $response = new RedirectResponse($this->router->generateUri('home'));
                break;
        }
        return $response;
    }

    /**
     * @param ServerRequestInterface $request
     */
    protected function sendActivationMail(ServerRequestInterface $request)
    {
        $identity = $this->auth->getIdentity();
        $this->mailer->sendActivationMail($identity, $request->getAttribute('layoutInfo')->getLang());
    }

    /**
     * @param ServerRequestInterface $request
     */
    private function requestActivateProtection(ServerRequestInterface $request)
    {
        $data = $request->getParsedBody();
        if (isset($data['name']) && isset($data['phone'])) {
            $identity = $this->auth->getIdentity();

            $this->mailer->sendActivateProtectionRequest(
                $data['name'],
                $data['phone'],
                $identity->getEmail(),
                $identity->getUuid()
            );
        }
    }
}