<?php

namespace App\Action;

use App\Authentication\UserService;
use App\Authentication\SupportService;
use App\Traits\IsXmlHttpRequest;
use Interop\Container\ContainerInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Expressive\Router\RouterInterface;
use Zend\Expressive\Template\TemplateRendererInterface;
use Zend\Expressive\ZendView\ZendViewRenderer;
use App\Authentication\Tokenizer\Authentication;
use Dflydev\FigCookies\SetCookie;
use Dflydev\FigCookies\SetCookies;
use Zend\Diactoros\Response\JsonResponse;
use Zend\Hydrator\DelegatingHydrator;

/**
 * Class Common
 * @package App\Action
 */
abstract class Common
{

    use IsXmlHttpRequest;

    /**
     * @var RouterInterface
     */
    protected $router;

    /**
     * @var ZendViewRenderer
     */
    protected $template;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * Common constructor.
     * @param RouterInterface $router
     * @param TemplateRendererInterface|null $template
     * @param ContainerInterface $container
     */
    public function __construct(RouterInterface $router, TemplateRendererInterface $template = null, ContainerInterface $container)
    {
        $this->router   = $router;
        $this->template = $template;
        $this->container = $container;
        $container->get('layout')->setTemplate('layout/default');

    }

    abstract function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null);

    /**
     * @param ServerRequestInterface $request
     */
    protected function initInfo(ServerRequestInterface $request)
    {

    }

    /**
     * @param $data
     * @param $request
     * @return ResponseInterface|JsonResponse
     */
    protected function authenticate($data, $request)
    {
        
        /** @var UserService $auths */
        $isValidUser = false;
        $auths = $this->container->get(UserService::class);
        $auths->setCredential($data);

        $result = $auths->authenticate();

        if ($result->isValid()) {
            $isValidUser = true;
        }
        
        if($isValidUser) {
            $token = $this->getToken($result->getIdentity());

            $tokenCookie = SetCookie::create($auths->getCookieTokenName())
                ->withExpires($data['remember_pass'] ? new \DateTime('+3 month') : 0)
                ->withMaxAge(0)
                ->withDomain($request->getUri()->getHost())
                ->withPath('/')
                ->withValue($token);

            $response = new JsonResponse([                
                'result' => true, 
                'type' => $isValidSupport ? 'support' : 'user',
                'role' => $result->getIdentity()->getRoleId(),
                'token' => (string) $tokenCookie
            ]);

            /** @var SetCookies $cookies */
            $cookies = SetCookies::fromResponse($response);
            $cookies = $cookies->with($tokenCookie);

            $response = $cookies->renderIntoSetCookieHeader($response);            
        } else {
            $messages = $result->getMessages();
            $response = new JsonResponse(['result' => false, 'msg' => reset($messages)]);
            $response->withStatus(502, $result->getMessages());
        }
        
        return $response;
    }

    /**
     * @param $identity
     * @return string
     */
    protected function getToken($identity)
    {
        /** @var Authentication $tokenizer */
        $tokenizer = $this->container->get(Authentication::class);
        /** @var DelegatingHydrator $hydrator */
        $hydrator = $this->container->get(DelegatingHydrator::class);

        return $tokenizer->encode($hydrator->extract($identity));

    }

}