<?php
/**
 * Copyright (c) 2017 Serhii Borodai <clarifying@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *
 */

namespace App\Action\User;


use App\Action\Common;
use App\Authentication\AggregateAuth;
use App\Authentication\SupportService;
use App\Authentication\Tokenizer\Authentication;
use App\Authentication\Tokenizer\Confirmation;
use App\Authentication\UserService;
use App\Entity\User;
use App\Model\Users;
use Firebase\JWT\ExpiredException;
use Monolog\Logger;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Slim\Flash\Messages;
use Zend\Diactoros\Response\HtmlResponse;
use Zend\Diactoros\Response\RedirectResponse;
use Zend\Hydrator\DelegatingHydrator;

/**
 * Class Activate
 * @package App\Action\User
 */
class Activate extends Common
{

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param callable|null $next
     * @return HtmlResponse
     */
    public function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null)
    {
        /** @var AggregateAuth $auth */
        $auth = new AggregateAuth($this->container->get(UserService::class), $this->container->get(SupportService::class));

        /** @var Authentication $tokenizer */
        $tokenizer = $this->container->get(Confirmation::class);

        $token = $request->getAttribute('token');
        /** @var Users $usersModel */
        $usersModel = $this->container->get(Users::class);

        /** @var Messages $flash */
        $flash = $request->getAttribute('flash');
        try {
            $decoded = $tokenizer->decode($token);
            $id = $decoded->data->id;

            /** @var User $user */
            $user = $usersModel->findById($id);
            $user->setActivated(true);
            $usersModel->save($user);
            
            // on success activation redirect to user cabinet
            $uri = $this->router->generateUri('user.cabinet', [
                'lang' => $request->getAttribute('layoutInfo')->getLang(),
            ]);

            $response = new RedirectResponse($uri);
            
        } catch (ExpiredException $e) {

            /** @var DelegatingHydrator $hyd */
            $hyd = $this->container->get(DelegatingHydrator::class);
            $user = $usersModel->findById($auth->getIdentity()->getId());

            $response = new HtmlResponse($this->template->render('app::user/expired-activation', [
                'lang' => $request->getAttribute('layoutInfo')->getLang(),
                'activationToken' => $tokenizer->encode($hyd->extract($user)),
            ]));
        } catch (\Exception $e) {
            /** @var Messages $flash */
            $flash = $request->getAttribute('flash');
            $flash->addMessage('error', $e->getMessage());

            /** @var Logger $log */
            $log = $this->container->get(Logger::class);
            $log->error($e->getMessage(), ['trace' => $e->getTraceAsString()]);

            $response = new RedirectResponse('/');
        }

        return $response;
    }
}